//
// ViewController.swift
// Cars DataGrid Sample
//
// This sample introduces how to create UIDataGrid object and  add corresponding DataGrid
// Columns that contain Picker, DatePicker,  TextField input box, Label Column,  TextView
// editor, Button Column, and other column types into a UIDataGrid on your iOS app layout.
//
// The main idea of the samples is to provide you with  a suggestion how to update values
// in other DataGrid  cells at  a row when  the current  column  cell has  been updated.
// For this case, when we need to modify corresponding cells we are implementing specific
// DataGrid Columns' value changing events.
//
// Also the code example gives hint how to deal with source data when you need manipulate
// and update a set of cells/rows in the UIDataGrid interface  and store the updates back
// into the datasource mutable array object. It explains how to enforce that by using the
// updateDataSource property of UIDataGrid object.
//
// Please read on-line documentation: http://rustemsoft.com/TouchControlsDoc/index.html
//
// Copyright (c) 2015 Rustemsoft LLC. All rights reserved.
//
// ======================================================================================
// Xcode 6.1.1 still has problems related to framework build and referencing.
// We have listed resolutions for couple of possible issues you may encounter:
//
// 1. When you run the Sample project for very first time the build may fail. To fix that
//   remove the reference to TouchControlsTrial.framework and re-reference to the library
//   as following:
// - In the Project Navigator, select the Sample project.
// - Select the target.
// - Select the "General" tab.
// - Open "Embedded Binaries" expander.
// - Click the + button.
// - Select the TouchControlsTrial.framework.
// - Window -> Organizer -> Delete Derived Data and then: Product -> Clean.
//
// 2. Framework Resources, especially images, can be not visible on your app layout.
//   To fix that make a reference to TouchControls_Resources folder as following:
// - In the Project Navigator, select your project main folder.
// - Right click and select "Add Files to..."
// - Select the TouchControls_Resources folder and add it.
// =======================================================================================
//
// How to Manually Add a Swift Bridging Header?
//
// 1. Add a header file to your project, named:  MyProjectName-Bridging-Header.h.  This will be the
// single header file where you import any Objective-C code you want your Swift code to have access.
//
// 2. In your project build settings, find Swift Compiler – Code Generation, and next to Objective-C
// Bridging Header add the path to your bridging header file, from the project’s root folder.  So it
// could by MyProject/MyProject-Bridging-Header.h or simply MyProject-Bridging-Header.h  if the file
// lives in the project root folder.
//
// You only need one Bridging Header. Add your #import statements to this file,and your classes will
// now be available in your Swift code without any extra import statements.
//

import UIKit

// Device detection. Define size coefficient if it is iPhone or iPad
var COEFF_PAD: CGFloat { return ((UIDevice.currentDevice().userInterfaceIdiom == .Pad) ? 2.0 : 1.0) }

// Define public variables
var dgrid = UIDataGrid()
var btnPickerColumnSample = UIButton()
var btnDatePickerColumnSample = UIButton()
var btnTextViewColumnSample = UIButton()
var btnChange = UIButton()
var bottomView = UIView()
var lblSampleTitle = UILabel()

class ViewController: UIViewController, UIScrollViewDelegate, DataGridDelegate, UIAlertViewDelegate {

    override func viewDidLoad() {
        super.viewDidLoad()
    
        // Set background image
        var img = UIImage(named: "background.png")
        var bgImg = UIImageView(image: img)
        bgImg.frame = CGRect(x: 0, y: 0, width: self.view.frame.width, height: self.view.frame.height)
        self.view.addSubview(bgImg)
        
        // Create UIDataGrid object
        dgrid = UIDataGrid(frame: CGRect(x: COEFF_PAD * 10,  y: COEFF_PAD * 40, width: self.view.frame.size.width - COEFF_PAD * 20, height: self.view.frame.size.height - COEFF_PAD * 80))
        dgrid.delegate = self // set UIDataGrid's delegate to self (otherwise events will not work)
        
        // Define UIDataGrid's DataSource
        // 1. NSMutableArray DataSource
        //dgrid.DataSource = self.generateNSMutableArrayDataSource()
        
        // 2. Plist file DataSource
        dgrid.DataSource = NSMutableArray(contentsOfFile: NSBundle.mainBundle().pathForResource("Cars", ofType: "plist")!)
        
        // 3. JSON file DataSource
        //dgrid.DataSource = self.generateJSONfileDataSource()
        
        // 4. XML file DataSource
        //dgrid.DataSource = self.generateXMLfileDataSource()
        
        // Add UIDataGrid object into app's main view layout
        self.view.addSubview(dgrid)
        
        // Add bottomView panel where buttons will be hosted
        bottomView = UIView(frame: CGRect(x: 0, y: self.view.frame.size.height - COEFF_PAD * 30, width: self.view.frame.size.width, height: COEFF_PAD * 30))
        bottomView.backgroundColor = UIColor(red:0.2, green:0.4, blue:0.6, alpha:0.8)
        self.view.addSubview(bottomView)
        
        // Add btnPickerColumnSample button
        btnPickerColumnSample = UIButton(frame: CGRect(x: COEFF_PAD * 5, y: COEFF_PAD * 8, width: COEFF_PAD * 66, height: COEFF_PAD * 14))
        btnPickerColumnSample.setTitleColor(UIColor.redColor(), forState: UIControlState.Selected)
        btnPickerColumnSample.setTitleColor(UIColor(red:0.1, green:0.1, blue:0.19, alpha:1), forState:UIControlState.Normal)
        btnPickerColumnSample.backgroundColor = UIColor(red:1, green:1, blue:1, alpha:0.5)
        btnPickerColumnSample.setTitle("Picker Column Sample", forState: UIControlState.Normal)
        btnPickerColumnSample.titleLabel?.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
        btnPickerColumnSample.addTarget(self, action: "btnPickerColumnSampleClicked:", forControlEvents: .TouchUpInside)
        btnPickerColumnSample.selected = true
        bottomView.addSubview(btnPickerColumnSample)
        
        // Add btnDatePickerColumnSample button
        btnDatePickerColumnSample = UIButton(frame: CGRectMake(COEFF_PAD * 75, COEFF_PAD * 8, COEFF_PAD * 75, COEFF_PAD * 14))
        btnDatePickerColumnSample.setTitleColor(UIColor.redColor(), forState: UIControlState.Selected)
        btnDatePickerColumnSample.setTitleColor(UIColor(red:0.1, green:0.1, blue:0.19, alpha:1), forState: UIControlState.Normal)
        btnDatePickerColumnSample.backgroundColor = UIColor(red:1, green:1, blue:1, alpha:0.5)
        btnDatePickerColumnSample.setTitle("DatePicker Column Sample", forState: UIControlState.Normal)
        btnDatePickerColumnSample.titleLabel?.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
        btnDatePickerColumnSample.addTarget(self, action: "btnDatePickerColumnSampleClicked:", forControlEvents: .TouchUpInside)
        btnDatePickerColumnSample.selected = false
        bottomView.addSubview(btnDatePickerColumnSample)
        
        // Add btnTextViewColumnSample button
        btnTextViewColumnSample = UIButton(frame: CGRectMake(COEFF_PAD * 160, COEFF_PAD * 8, COEFF_PAD * 75, COEFF_PAD * 14))
        btnTextViewColumnSample.backgroundColor = UIColor(red:1, green:1, blue:1, alpha:0.5)
        btnTextViewColumnSample.setTitleColor(UIColor.redColor(), forState: UIControlState.Selected)
        btnTextViewColumnSample.setTitleColor(UIColor(red:0.1, green:0.1, blue:0.19, alpha:1), forState: UIControlState.Normal)
        btnTextViewColumnSample.setTitle("TextView Column Sample", forState: UIControlState.Normal)
        btnTextViewColumnSample.titleLabel?.font = UIFont(name:"Georgia", size:COEFF_PAD * 6)
        btnTextViewColumnSample.addTarget(self, action: "btnTextViewColumnSampleClicked:", forControlEvents: .TouchUpInside)
        btnTextViewColumnSample.selected = false
        bottomView.addSubview(btnTextViewColumnSample)
        
        // Add btnChange button
        btnChange = UIButton(frame: CGRectMake(self.view.frame.size.width - COEFF_PAD * 75, COEFF_PAD * 7, COEFF_PAD * 65, COEFF_PAD * 16))
        btnChange.titleLabel?.textColor = UIColor(red:0.95, green:0.1, blue:0.19, alpha:1)
        btnChange.backgroundColor = UIColor(red:1, green:1, blue:1, alpha:0.5)
        btnChange.setImage(UIImage(named:"change.gif"), forState: UIControlState.Normal)
        btnChange.titleLabel?.text = "Analyze"
        btnChange.titleLabel?.font = UIFont(name:"HelveticaNeue-CondensedBold", size:COEFF_PAD * 10)
        btnChange.layer.cornerRadius = 7
        btnChange.layer.borderColor = UIColor.whiteColor().CGColor
        btnChange.layer.borderWidth = COEFF_PAD * 0.2
        btnChange.addTarget(self, action: "btnChangeClicked:", forControlEvents: .TouchUpInside)
        bottomView.addSubview(btnChange)
        
        // Define Sample Title label
        lblSampleTitle = UILabel(frame: CGRectMake(COEFF_PAD * 10, COEFF_PAD * 30, COEFF_PAD * 120, COEFF_PAD * 10))
        lblSampleTitle.textColor = UIColor.purpleColor()
        lblSampleTitle.font = UIFont(name:"HelveticaNeue-CondensedBold", size:COEFF_PAD * 9)
        self.view.addSubview(lblSampleTitle)
        
        // Set up UIDataGrid object's layout to show Picker Column Sample
        self.btnPickerColumnSampleClicked(btnPickerColumnSample)
    }
    
    // rotate iPhone/iPad
    override func willRotateToInterfaceOrientation(toInterfaceOrientation: UIInterfaceOrientation, duration: NSTimeInterval) {
        // Define main screen width and height
        var mainWidth = CGFloat((toInterfaceOrientation == UIInterfaceOrientation.LandscapeLeft || toInterfaceOrientation == UIInterfaceOrientation.LandscapeRight) ? self.view.frame.size.height : self.view.frame.size.width)
        var mainHeight = CGFloat((toInterfaceOrientation == UIInterfaceOrientation.LandscapeLeft || toInterfaceOrientation == UIInterfaceOrientation.LandscapeRight) ? self.view.frame.size.width : self.view.frame.size.height)
        // Set btnChange x coordinate depending on interface orientation
        var originInX = CGFloat(mainWidth - btnChange.frame.size.width - COEFF_PAD * 10)
        // Set the btnChange's x offset
        bottomView.frame = CGRectMake(0, mainHeight - COEFF_PAD * 30, mainWidth, COEFF_PAD * 30)
        btnChange.frame = CGRectMake(originInX, COEFF_PAD * 7, btnChange.frame.size.width, btnChange.frame.size.height)
    }
    
    // Click btnPickerColumnSample button
    func btnPickerColumnSampleClicked(btn: UIButton) {
        // Reset buttons' 'Selected' status
        btnPickerColumnSample.selected = false
        btnDatePickerColumnSample.selected = false
        btnTextViewColumnSample.selected = false
        btn.selected = true
        lblSampleTitle.text = btn.titleLabel?.text
    
        // Set up UIDataGrid object's layout to show Picker Column Sample
        // Adding Columns
        var cols = NSMutableArray()
    
        // Define DataGridImageColumn object
        var colImg = DataGridImageColumn()
        colImg.dataFieldIndex = 8
        colImg.Title = "Logo"
        colImg.Width = COEFF_PAD * 50
        colImg.header.textAlignment = NSTextAlignment.Center
        colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40)
        colImg.cellLayoutPosition = LayoutPositions.RightCenter
        colImg.cellLayoutPositionX = COEFF_PAD * 5
        colImg.cellLayoutPositionY = COEFF_PAD * 10
        cols.addObject(colImg)
    
        // Define DataGridLabelColumn object
        var col = DataGridLabelColumn()
        col.dataFieldIndex = 1
        col.Width = COEFF_PAD * 60
        col.Title = "Corporation"
        col.backgroundColor = UIColor(red:0.01, green:0.01, blue:0.5, alpha:0.3)
        col.header.textAlignment = NSTextAlignment.Center
        col.HeaderBackgroundColor = UIColor(red:0.01, green:0.01, blue:0.5, alpha:0.3)
        col.HeaderForegroundColor = UIColor.blackColor()
        col.HeaderFont = UIFont(name:"Verdana", size:COEFF_PAD * 7)
        col.borderColor = UIColor.darkTextColor()
        col.borderWidth = COEFF_PAD * 0.3
        // Define a custom cell size and cell's position
        col.cellSize = CGSizeMake(COEFF_PAD * 52, COEFF_PAD * 16)
        col.cellLayoutPosition = LayoutPositions.Center
        col.cellTemplate.textAlignment = NSTextAlignment.Center
        col.cellTemplate.textColor = UIColor.darkTextColor()
        col.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        cols.addObject(col)
    
        // Define DataGridPickerColumn object
        var colpkr = DataGridPickerColumn()
        colpkr.dataFieldIndex = 1
        // Define Data source for picker items
        let dsr = NSMutableArray()
        dsr.addObject("")
        dsr.addObject("Ford")
        dsr.addObject("Mitsubishi")
        dsr.addObject("Toyota")
        dsr.addObject("Honda")
        dsr.addObject("Subaru")
        dsr.addObject("Dodge")
        dsr.addObject("Jeep")
        dsr.addObject("Volkswagen")
        dsr.addObject("Nissan")
        dsr.addObject("Chevrolet")
        dsr.addObject("Fiat")
        dsr.addObject("Hyundai")
        dsr.addObject("Kia")
        dsr.addObject("Mercedes")
        dsr.addObject("Peugeot")
        dsr.addObject("Renault")
        colpkr.pickerDataSource = dsr
        colpkr.Title = "Select"
        colpkr.hidden = false
        colpkr.pickerFrameColor = PickerFrameColors.greenPickerFrameColor
        colpkr.header.textAlignment = NSTextAlignment.Center
        colpkr.Width = COEFF_PAD * 120
        // Define a custom cell size
        colpkr.cellSize = CGSizeMake(COEFF_PAD * 100, COEFF_PAD * 60)
        // Define a custom UILabel object as cellLabelTemplate
        var lblRow = UILabel()
        // Assign Text Alignment
        lblRow.textAlignment = NSTextAlignment.Center
        lblRow.textColor = UIColor.redColor()
        lblRow.font = UIFont(name:"HelveticaNeue-CondensedBold", size:COEFF_PAD * 17)
        colpkr.cellLabelTemplate = lblRow
        colpkr.pickerRowHeight = COEFF_PAD * 25
        cols.addObject(colpkr)
    
        // Define DataGridSwitchColumn object
        var colSwh = DataGridSwitchColumn()
        colSwh.dataFieldIndex = 12
        colSwh.Title = "Access"
        colSwh.header.textAlignment = NSTextAlignment.Left
        colSwh.hidden = false
        colSwh.backgroundColor = UIColor(red:0.1, green:0.1, blue:0.1, alpha:0.1)
        colSwh.cellSize = CGSizeMake(COEFF_PAD * 50, COEFF_PAD * 30)
        colSwh.onTintColor = UIColor.greenColor()
        colSwh.borderWidth = 0
        cols.addObject(colSwh)
    
        // Define DataGridTextFieldColumn object
        var coltxt = DataGridTextFieldColumn()
        coltxt.dataFieldIndex = 6
        coltxt.Width = 115
        coltxt.Title = "Phone"
        coltxt.header.textAlignment = NSTextAlignment.Center
        coltxt.autocapitalizationType = UITextAutocapitalizationType.None
        coltxt.borderStyle = UITextBorderStyle.RoundedRect
        coltxt.clearButtonMode = UITextFieldViewMode.Never
        coltxt.contentVerticalAlignment = UIControlContentVerticalAlignment.Center
        coltxt.contentHorizontalAlignment = UIControlContentHorizontalAlignment.Fill
        coltxt.keyboardType = UIKeyboardType.PhonePad
        coltxt.returnKeyType = UIReturnKeyType.Done
        coltxt.TextAlignment = NSTextAlignment.Natural
        coltxt.textColor = UIColor.blackColor()
        coltxt.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 12)
        coltxt.recognizeCellTapped = false
        cols.addObject(coltxt)
    
        // Define DataGridButtonColumn object
        var colBtn = DataGridButtonColumn()
        colBtn.dataFieldIndex = 10
        colBtn.AutoSizeType = AutoSizeTypes.None
        colBtn.Title = "ZipCode"
        colBtn.font = UIFont(name:"Georgia-Italic", size:COEFF_PAD * 10)
        colBtn.header.textAlignment = NSTextAlignment.Center
        colBtn.cellSize = CGSizeMake(70, 40)
        colBtn.cellLayoutPosition = LayoutPositions.RightCenter
        colBtn.cellTemplate.layer.cornerRadius = 7
        colBtn.cellTemplate.layer.borderColor = UIColor.grayColor().CGColor
        colBtn.cellTemplate.layer.borderWidth = COEFF_PAD * 3.2
        colBtn.cellTemplate.layer.backgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2).CGColor
        cols.addObject(colBtn)
    
        // Add Columns to Column collection
        dgrid.Columns = cols
    
        // Assign UIDataGrid main properties
        dgrid.ColumnsWidth = COEFF_PAD * 110
        // Datagrid's main colors
        dgrid.backgroundColor = UIColor.clearColor()
        dgrid.foregroundColor = UIColor.purpleColor()
        // Color of every other row
        dgrid.alternatingBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.1)
        // Specify properties of headers
        dgrid.HeaderBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2)
        dgrid.HeaderForegroundColor = UIColor.whiteColor()
        dgrid.HeaderFont = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        dgrid.HeaderBorderColor = UIColor.lightGrayColor()
        dgrid.HeaderBorderWidth = COEFF_PAD * 0.3
        dgrid.HeaderScrollHide = false
    
        dgrid.rowBorderHeight = 1
        dgrid.rowBorderColor = UIColor.cyanColor()
        dgrid.RowHeight = COEFF_PAD * 70
        dgrid.datagridFont = UIFont(name:"Verdana", size:COEFF_PAD * 4)
        dgrid.updateDataSource = true
    
        // Bind to populate data from DataSource into UIDataGrid object
        dgrid.DataBind()
    }
    
    // Click btnDatePickerColumnSample button
    func btnDatePickerColumnSampleClicked(btn: UIButton) {
        // Reset buttons' 'Selected' status
        btnPickerColumnSample.selected = false
        btnDatePickerColumnSample.selected = false
        btnTextViewColumnSample.selected = false
        btn.selected = true
        lblSampleTitle.text = btn.titleLabel?.text
    
        // Set up UIDataGrid object's layout to show DatePicker Column Sample
        // Adding Columns
        var cols = NSMutableArray()
    
        // Define DataGridImageColumn object
        var colImg = DataGridImageColumn()
        colImg.dataFieldIndex = 8
        colImg.Title = "Logo"
        colImg.Width = COEFF_PAD * 50
        colImg.header.textAlignment = NSTextAlignment.Center
        colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40)
        colImg.cellLayoutPosition = LayoutPositions.RightCenter
        colImg.cellLayoutPositionX = COEFF_PAD * 5
        colImg.cellLayoutPositionY = COEFF_PAD * 10
        cols.addObject(colImg)
    
        // Define DataGridLabelColumn object
        var col = DataGridLabelColumn()
        col.dataFieldIndex = 11
        col.Width = COEFF_PAD * 30
        col.Title = "%%"
        col.header.textAlignment = NSTextAlignment.Center
        col.cellTemplate.textAlignment = NSTextAlignment.Center
        col.cellTemplate.textColor = UIColor.darkTextColor()
        col.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        cols.addObject(col)
    
        // Define DataGridSliderColumn object
        var colSldr = DataGridSliderColumn()
        colSldr.dataFieldIndex = 11
        colSldr.Title = "Availability"
        colSldr.header.textAlignment = NSTextAlignment.Center
        colSldr.hidden = false
        colSldr.Width = COEFF_PAD * 90
        colSldr.cellSize = CGSizeMake(COEFF_PAD * 80, COEFF_PAD * 50)
        colSldr.backgroundColor = UIColor(red:0.1, green:0.1, blue:0.1, alpha:0.1)
        colSldr.borderWidth = 0
        colSldr.minimumValue = 0
        colSldr.maximumValue = 110
        colSldr.cellTemplate.tintColor = UIColor.redColor()
        cols.addObject(colSldr)
    
        // Define DataGridDatePickerColumn object
        var colDtPkr = DataGridDatePickerColumn()
        colDtPkr.dataFieldIndex = 5
        colDtPkr.Width = COEFF_PAD * 280
        colDtPkr.Title = "Date"
        colSldr.header.textAlignment = NSTextAlignment.Center
        colDtPkr.hidden = false
        colDtPkr.pickerFrameColor = PickerFrameColors.bluePickerFrameColor
        colDtPkr.cellTemplate.datePickerMode = UIDatePickerMode.Date
        colDtPkr.dateFormat = "MM/dd/yyyy"
        cols.addObject(colDtPkr)
    
        // Define DataGridButtonColumn object
        var colBtn = DataGridButtonColumn()
        colBtn.dataFieldIndex = 10
        colBtn.AutoSizeType = AutoSizeTypes.None
        colBtn.Title = "Press to Start!"
        colBtn.font = UIFont(name:"Georgia", size:COEFF_PAD * 10)
        colBtn.header.textAlignment = NSTextAlignment.Center
        colBtn.cellSize = CGSizeMake(COEFF_PAD * 70, COEFF_PAD * 40)
        colBtn.cellTemplate.layer.cornerRadius = 7
        colBtn.cellTemplate.layer.borderColor = UIColor(red:0.2, green:0.3, blue:0.2, alpha:0.2).CGColor
        colBtn.cellTemplate.layer.borderWidth = COEFF_PAD * 3.2
        colBtn.cellTemplate.layer.backgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2).CGColor
        cols.addObject(colBtn)
    
        // Add Columns to Column collection
        dgrid.Columns = cols
    
        // Assign UIDataGrid main properties
        dgrid.ColumnsWidth = COEFF_PAD * 110
        // Datagrid's main colors
        dgrid.backgroundColor = UIColor(red:0.9, green:0.9, blue:0.9, alpha:0.3)
        dgrid.foregroundColor = UIColor.purpleColor()
        // Color of every other row
        dgrid.alternatingBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2)
        // Specify properties of headers
        dgrid.HeaderBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2)
        dgrid.HeaderForegroundColor = UIColor.whiteColor()
        dgrid.HeaderFont = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        dgrid.HeaderBorderColor = UIColor.lightGrayColor()
        dgrid.HeaderBorderWidth = COEFF_PAD * 0.3
        dgrid.HeaderScrollHide = false
    
        dgrid.cellBorderColor = UIColor.cyanColor()
        dgrid.cellBorderWidth = 0.5
        dgrid.rowBorderColor = UIColor.clearColor()
        dgrid.rowBorderHeight = COEFF_PAD * 1.5
        dgrid.RowHeight = COEFF_PAD * 70
        dgrid.datagridFont = UIFont(name:"Verdana", size:COEFF_PAD * 4)
        dgrid.updateDataSource = true
    
        // Bind to populate data from DataSource into UIDataGrid object
        dgrid.DataBind()
    }
    
    // Click btnTextViewColumnSample button
    func btnTextViewColumnSampleClicked(btn: UIButton) {
        // Reset buttons' 'Selected' status
        btnPickerColumnSample.selected = false
        btnDatePickerColumnSample.selected = false
        btnTextViewColumnSample.selected = false
        btn.selected = true
        lblSampleTitle.text = btn.titleLabel?.text
    
        // Set up UIDataGrid object's layout to show TextView Column Sample
        // Adding Columns
        var cols = NSMutableArray()
    
        // Define DataGridImageColumn object
        var colImg = DataGridImageColumn()
        colImg.dataFieldIndex = 8
        colImg.Title = "Logo"
        colImg.Width = COEFF_PAD * 50
        colImg.header.textAlignment = NSTextAlignment.Center
        colImg.cellSize = CGSizeMake(COEFF_PAD * 40, COEFF_PAD * 40)
        colImg.cellLayoutPosition = LayoutPositions.RightCenter
        colImg.cellLayoutPositionX = COEFF_PAD * 5
        colImg.cellLayoutPositionY = COEFF_PAD * 10
        cols.addObject(colImg)
    
        // Define DataGridLabelColumn object
        var col = DataGridLabelColumn()
        col.dataFieldIndex = 1
        col.AutoSizeType = AutoSizeTypes.CellContentSize
        col.Title = "Corporation"
        col.header.textAlignment = NSTextAlignment.Center
        col.cellTemplate.textAlignment = NSTextAlignment.Center
        col.cellTemplate.textColor = UIColor.darkTextColor()
        col.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        cols.addObject(col)
    
        // Define DataGridTextViewColumn object
        var cTxtVw = DataGridTextViewColumn()
        cTxtVw.dataFieldIndex = 9
        cTxtVw.Title = "Remark"
        cTxtVw.header.textAlignment = NSTextAlignment.Center
        cTxtVw.autocapitalizationType = UITextAutocapitalizationType.None
        cTxtVw.keyboardType = UIKeyboardType.ASCIICapable
        cTxtVw.returnKeyType = UIReturnKeyType.Done
        cTxtVw.TextAlignment = NSTextAlignment.Natural
        cTxtVw.textColor = UIColor.blackColor()
        cTxtVw.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        cTxtVw.recognizeCellTapped = false
        cTxtVw.Width = COEFF_PAD * 250
        cTxtVw.cellSize = CGSizeMake(COEFF_PAD * 230, COEFF_PAD * 70)
        cTxtVw.cellLayoutPositionX = COEFF_PAD * 5
        cTxtVw.cellLayoutPositionY = COEFF_PAD * 10
        cols.addObject(cTxtVw)
    
        // Define DataGridTextFieldColumn object
        var coltxt = DataGridTextFieldColumn()
        coltxt.dataFieldIndex = 12
        coltxt.Title = "Number"
        // Format numeric input
        coltxt.numberFormatterStyle = NSNumberFormatterStyle.DecimalStyle
        coltxt.numberWithType = numberWithTypes.Float
        // Header's title position
        coltxt.header.textAlignment = NSTextAlignment.Center
        coltxt.autocapitalizationType = UITextAutocapitalizationType.None
        coltxt.borderStyle = UITextBorderStyle.RoundedRect
        coltxt.clearButtonMode = UITextFieldViewMode.Never
        coltxt.contentVerticalAlignment = UIControlContentVerticalAlignment.Center
        coltxt.contentHorizontalAlignment = UIControlContentHorizontalAlignment.Fill
        coltxt.keyboardType = UIKeyboardType.NumbersAndPunctuation
        coltxt.returnKeyType = UIReturnKeyType.Done
        coltxt.TextAlignment = NSTextAlignment.Natural
        coltxt.textColor = UIColor.blackColor()
        coltxt.cellTemplate.font = UIFont(name:"Verdana", size:COEFF_PAD * 12)
        coltxt.recognizeCellTapped = false
        cols.addObject(coltxt)
    
        // Add Columns to Column collection
        dgrid.Columns = cols
    
        // Assign UIDataGrid main properties
        dgrid.ColumnsWidth = COEFF_PAD * 110
        // Datagrid's main colors
        dgrid.backgroundColor = UIColor.clearColor()
        dgrid.foregroundColor = UIColor.purpleColor()
        // Color of every other row
        dgrid.alternatingBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.1)
        // Specify properties of headers
        dgrid.HeaderBackgroundColor = UIColor(red:0.1, green:0.2, blue:0.3, alpha:0.2)
        dgrid.HeaderForegroundColor = UIColor.whiteColor()
        dgrid.HeaderFont = UIFont(name:"Verdana", size:COEFF_PAD * 8)
        dgrid.HeaderBorderColor = UIColor.lightGrayColor()
        dgrid.HeaderBorderWidth = COEFF_PAD * 0.3
        dgrid.HeaderScrollHide = false
    
        dgrid.columnBorderColor = UIColor.cyanColor()
        dgrid.columnBorderWidth = COEFF_PAD * 0.5
        dgrid.RowHeight = COEFF_PAD * 90
        dgrid.datagridFont = UIFont(name:"Verdana", size:COEFF_PAD * 4)
        dgrid.updateDataSource = true
    
        // Bind to populate data from DataSource into UIDataGrid object
        dgrid.DataBind()
    }
    
    // Click btnChange button
    func btnChangeClicked(btn: UIButton) {
        // Change a particular cell of a particular column
        // Change second DataGridImageCell in very first column
        var colImg = dgrid.Columns.objectAtIndex(0) as DataGridImageColumn
        var cellImg = colImg.Cells.objectAtIndex(1) as DataGridImageCell
        var changed: Bool = CGColorEqualToColor(cellImg.image.layer.borderColor, UIColor.redColor().CGColor)
        if changed {
            colImg.header.font = dgrid.HeaderFont
            colImg.header.backgroundColor = colImg.HeaderBackgroundColor
            colImg.header.textColor = UIColor.blueColor()
            cellImg.image.layer.borderColor = colImg.borderColor.CGColor
            cellImg.image.layer.borderWidth = 0
        } else {
            colImg.header.font = UIFont(name:"Verdana", size:COEFF_PAD * 10)
            colImg.header.backgroundColor = UIColor(red:0.6, green:0.1, blue:0.2, alpha:0.9)
            cellImg.image.layer.borderColor = UIColor.redColor().CGColor
            cellImg.image.layer.borderWidth = COEFF_PAD * 3
        }
    
        // Change second DataGridLabelCell in the second column
        var col = dgrid.Columns.objectAtIndex(1) as DataGridLabelColumn
        var cell = col.Cells.objectAtIndex(1) as DataGridLabelCell
        if changed {
            cell.label.backgroundColor = dgrid.backgroundColor
            cell.label.textColor = col.textColor
            cell.label.textAlignment = col.cellTemplate.textAlignment
        } else {
            cell.label.backgroundColor = UIColor.lightGrayColor()
            cell.label.textColor = UIColor.blueColor()
            cell.label.textAlignment = NSTextAlignment.Center
        }
    
        // Depending on what Sample is loaded change corresponding cells
        if btnPickerColumnSample.selected {
            var colPkr = dgrid.Columns.objectAtIndex(2) as DataGridPickerColumn
            var cellPkr = colPkr.Cells.objectAtIndex(0) as DataGridPickerCell
            if changed {
                cellPkr.backgroundColor = dgrid.backgroundColor
            } else {
                cellPkr.backgroundColor = UIColor.yellowColor()
            }
        } else if btnDatePickerColumnSample.selected {
            var colSld = dgrid.Columns.objectAtIndex(2) as DataGridSliderColumn
            var cellSld = colSld.Cells.objectAtIndex(0) as DataGridSliderCell
            if changed {
                cellSld.slider.tintColor = colSld.tintColor
            } else {
                cellSld.slider.tintColor = UIColor.yellowColor()
            }
            var colPkr = dgrid.Columns.objectAtIndex(3) as DataGridDatePickerColumn
            var cellPkr = colPkr.Cells.objectAtIndex(0) as DataGridDatePickerCell
            if changed {
                cellPkr.backgroundColor = dgrid.backgroundColor
            } else {
                cellPkr.backgroundColor = UIColor.yellowColor()
            }
        } else if btnTextViewColumnSample.selected {
            var col = dgrid.Columns.objectAtIndex(2) as DataGridTextViewColumn
            var cell = col.Cells.objectAtIndex(0) as DataGridTextViewCell
            if changed {
                cell.textView.backgroundColor = col.backgroundColor
                cell.textView.textColor = col.textColor
            } else {
                cell.textView.backgroundColor = UIColor.yellowColor()
                cell.textView.textColor = UIColor.redColor()
            }
        }
        return
    }
    
    // Event-handling procedure that fires when a datagrid cell is tapped
    func datagridCellTapped(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, value: NSString) {
        var msg: NSString = "Value:" + value
        var ttl: NSString = "Column:" + String(colIndex) + "\rRow:" + String(rowIndex)
        // Show a message
        self.msgbox(msg, title: ttl)
        return
    }
    
    // Event-handling procedure that fires when a datagrid column's title is tapped
    func datagridColumnTitleTapped(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, title: NSString) {
        // Show a message
        self.msgbox("Title: " + title, title: "Column: " + String(colIndex))
        return
    }
    
    // Event-handling procedure that fires when a datagrid TextField Column cell's value is changed after editing
    func datagridTextFieldColumnCellEditingChanged(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender:UITextField) {
        // Identify sender's DataGridTextFieldColumn
        var col = dgrid.Columns.objectAtIndex(colIndex) as DataGridTextFieldColumn
        // Phone number formatting (works for 'Phone' field only)
        if col.Title == "Phone" { sender.text = self.phoneNumberFormat(sender.text) }
        return
    }

    // Event-handling procedure that fires when a datagrid TextField Column cell's starts editing
    func datagridTextFieldColumnCellEditingBegin(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender:UITextField) {
        //==========
        // Put some commands here.
        //==========
        return
    }
    
    // Event-handling procedure that fires when a datagrid TextField Column cell's ends editing
    func datagridTextFieldColumnCellEditingEnd(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender:UITextField) {
        //==========
        // Put some commands here.
        //==========
        return
    }
    
    // Event-handling procedure that fires when a datagrid TextView Column cell's value is changed
    func datagridTextViewColumnCellEditingChanged(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender:UITextView) {
        // Identify sender's DataGridTextViewColumn
        var col = dgrid.Columns.objectAtIndex(colIndex) as DataGridTextViewColumn
        // Identify index of datasource's data field that connected to the DataGridTextViewColumn
        var dataFieldIndex = Int(col.dataFieldIndex)
        // Locate initial cell's data value in datagrid's data source
        var itm = NSMutableArray()
        itm = dgrid.DataSource.objectAtIndex(rowIndex) as NSMutableArray
        var initValue = itm.objectAtIndex(dataFieldIndex) as NSString
        // Show a message
        self.msgbox("Initial:" + String(initValue) + "\rChanged:" + sender.text, title: "Column:" + String(colIndex))
        return
    }
    
    // Event-handling procedure that fires when a datagrid Switch Column cell's value is changed
    func datagridSwitchColumnCellValueChanged(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender:UISwitch) {
        // Show switch cell changed message
        self.msgbox("Row: " + String(rowIndex) + "\rChanged to: " + (sender.on ? "YES" : "NO"), title: "Switch Column: " + String(colIndex))
        return
    }

    // Procedure that draws every item in each PickerColumn cell populating from pickerDataSource
    func datagridPickerColumnCellViewForRow(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, rowInPicker:NSInteger, component: NSInteger, reusingView: UIView, sender: UIPickerView) -> UIView {
        // Define a custom UILabel object
        var lblRow = UILabel()
        // Identify the picker column object
        var col = dgrid.Columns.objectAtIndex(colIndex) as DataGridPickerColumn
        lblRow.frame = sender.frame
        lblRow.textAlignment = NSTextAlignment.Center
        // Define the picker cell content
        var rowItem = col.pickerDataSource.objectAtIndex(rowInPicker) as NSString
        lblRow.textColor = UIColor.purpleColor()
        // Add the text
        lblRow.text = rowItem
        return lblRow
    }

    // Procedure that runs every time when an item in a PickerColumn cell has been selected
    func datagridPickerColumnCellRowSelected(datagrid: UIDataGrid!, rowIndex: Int, colIndex: Int, rowInPicker row: Int, component: Int, sender pickerView: UIPickerView!) {
        var col = dgrid.Columns.objectAtIndex(colIndex) as DataGridPickerColumn
        var itm = dgrid.DataSource.objectAtIndex(rowIndex) as NSMutableArray
        var st = itm.objectAtIndex(col.dataFieldIndex) as NSString
        var str = st.lowercaseString + ".jpg"
        var colImg = dgrid.Columns.objectAtIndex(0) as DataGridImageColumn
        var cellImg = colImg.Cells.objectAtIndex(rowIndex) as DataGridImageCell
        var imgRect = CGRectMake(cellImg.image.frame.origin.x, cellImg.image.frame.origin.y, cellImg.image.frame.size.width, cellImg.image.frame.size.height)
        cellImg.image.removeFromSuperview()
        var img = UIImageView(image: UIImage(named: str))
        img.frame = imgRect
        cellImg.addSubview(img)
        var colLbl = dgrid.Columns.objectAtIndex(1) as DataGridLabelColumn
        var cellLbl = colLbl.Cells.objectAtIndex(rowIndex) as DataGridLabelCell
        cellLbl.label.text = st
        return
    }
    
    // Event-handling procedure that fires when a datagrid Slider Column cell's value is changed
    func datagridSliderColumnCellValueChanged(datagrid: UIDataGrid, rowIndex: Int, colIndex: Int, sender: UISlider) {
        // Identify the Label column object
        var colLbl = dgrid.Columns.objectAtIndex(1) as DataGridLabelColumn
        // Identify the Label cell object
        var cellLbl = colLbl.Cells.objectAtIndex(rowIndex) as DataGridLabelCell
        // Change the Label's text with Slider's value
        cellLbl.label.text = sender.value.description
        return
    }
    
    // Event-handling procedure that fires when datagrid scrolling started
    func datagridScrollViewDidScroll(datagrid: UIDataGrid, scrollView: UIScrollView) {
        //==========
        // Put some commands here.
        //==========
        return
    }
    
    // Generate UIDataGrid object's DataSources
    // 1. NSMutableArray DataSource
    func generateNSMutableArrayDataSource() -> NSMutableArray {
    var dsource = NSMutableArray()
    var itm = NSMutableArray()
    itm.addObject("1")
    itm.addObject("Mitsubishi")
    itm.addObject("1")
    itm.addObject("3")
    itm.addObject("2/20/2004")
    itm.addObject("3/6/2004")
    itm.addObject("000-000-0000")
    itm.addObject("true")
    itm.addObject("mitsubishi.jpg")
    itm.addObject("The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night.")
    itm.addObject("100010001")
    itm.addObject("100")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("2")
    itm.addObject("Dodge")
    itm.addObject("2")
    itm.addObject("17")
    itm.addObject("2/20/2004")
    itm.addObject("3/6/2004")
    itm.addObject("111-111-1111")
    itm.addObject("false")
    itm.addObject("dodge.jpg")
    itm.addObject("Anti-Lock Brakes Air Bags AM/FM Stereo Radio Automatic Transmission Air Conditioning 4 or 6 Cylinder Engine Power-Assist Steering Tinted Windows Dual Mirrors Tilt Steering Wheel Power Top Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment) a bar with lighting a wash-stand and shelves. For creation of comfort in the compartments they are isolated from the salon by one common and two personal doors. Moreover passengers space in the carriage there are 2 washing-rooms with cold and hot water kitchen a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night.")
    itm.addObject("000200002")
    itm.addObject("10")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("3")
    itm.addObject("Toyota")
    itm.addObject("3")
    itm.addObject("40")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("222-222-2222")
    itm.addObject("true")
    itm.addObject("toyota.jpg")
    itm.addObject("The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night.")
    itm.addObject("123451234")
    itm.addObject("50")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("4")
    itm.addObject("Honda")
    itm.addObject("1")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("honda.jpg")
    itm.addObject("Anti-Lock Brakes Air Bags AM/FM Stereo Radio Automatic Transmission Air Conditioning 4 or 6 Cylinder Engine Power-Assist Steering Tinted Windows Dual Mirrors Tilt Steering Wheel Power Top Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment) a bar with lighting a wash-stand and shelves. For creation of comfort in the compartments they are isolated from the salon by one common and two personal doors. Moreover passengers space in the carriage there are 2 washing-rooms with cold and hot water kitchen a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night.")
    itm.addObject("123451234")
    itm.addObject("99")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("5")
    itm.addObject("Jeep")
    itm.addObject("2")
    itm.addObject("59")
    itm.addObject("")
    itm.addObject("3/6/2004")
    itm.addObject("")
    itm.addObject("true")
    itm.addObject("jeep.jpg")
    itm.addObject("The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night.")
    itm.addObject("")
    itm.addObject("90")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("6")
    itm.addObject("Subaru")
    itm.addObject("1")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("subaru.jpg")
    itm.addObject("Anti-Lock Brakes Air Bags AM/FM Stereo Radio Automatic Transmission Air Conditioning 4 or 6 Cylinder Engine Power-Assist Steering Tinted Windows Dual Mirrors Tilt Steering Wheel Power Top Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment) a bar with lighting a wash-stand and shelves. For creation of comfort in the compartments they are isolated from the salon by one common and two personal doors. Moreover passengers space in the carriage there are 2 washing-rooms with cold and hot water kitchen a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night.")
    itm.addObject("")
    itm.addObject("80")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("7")
    itm.addObject("Volkswagen")
    itm.addObject("1")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("true")
    itm.addObject("volkswagen.JPG")
    itm.addObject("The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night.")
    itm.addObject("")
    itm.addObject("70")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("8")
    itm.addObject("Ford")
    itm.addObject("2")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject(")")
    itm.addObject(")")
    itm.addObject(")false")
    itm.addObject("ford.JPG")
    itm.addObject("Anti-Lock Brakes Air Bags AM/FM Stereo Radio Automatic Transmission Air Conditioning 4 or 6 Cylinder Engine Power-Assist Steering Tinted Windows Dual Mirrors Tilt Steering Wheel Power Top Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the salon is 25 sq. metres. There are 2 tables. 2 soft extensible sofas and 6-8 arm-chairs. At passengers' serves there is I large compartment (6 sq. metres) and I small compartment (3 sq. metres). In each compartment there is a soft sofa (extensible sofa in the large compartment) a bar with lighting a wash-stand and shelves. For creation of comfort in the compartments they are isolated from the salon by one common and two personal doors. Moreover passengers space in the carriage there are 2 washing-rooms with cold and hot water kitchen a staff compartment and a service compartment. In the service compartment a refrigerator (volume 40 litres) for food and drinks store is intended. Electrical equipment is counted on connection with audio video. computer and other apparatuses. Qualified guards-stuards are at passengers' service during all day and night.")
    itm.addObject("999999999")
    itm.addObject("60")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("9")
    itm.addObject("Nissan")
    itm.addObject("3")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("true")
    itm.addObject("nissan.jpg")
    itm.addObject("The car is intended for passenger transportation in separate compartments. At passengers' service the car has 9 four-berth compartments. In each compartment there are two sofas two sleeping shelves and a table. Compartments are separated from the corridor by sliding doors. Moreover the passenger space in the car there are two wash-rooms with wash-basins a staff compartment and a service compartment. A refrigerator (volume 40 litres) for food and drinks store is intended in the service compartment. Qualified guards-stuards are at passengers' service all day and night.")
    itm.addObject("")
    itm.addObject("50")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("10")
    itm.addObject("Chevrolet")
    itm.addObject("1")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("chevrolet.jpg")
    itm.addObject("Anti-Lock Brakes Air Bags AM/FM Stereo Radio Automatic Transmission Air Conditioning 4 or 6 Cylinder Engine Power-Assist Steering Tinted Windows Dual Mirrors Tilt Steering Wheel Power Top Power Door Locks. The carriage is intended for passengers rest in the salon and for extra-comfortable compartments. Total square of the")
    itm.addObject("")
    itm.addObject("40")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("11")
    itm.addObject("Fiat")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("fiat.jpg")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("30")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("12")
    itm.addObject("Hyundai")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("hyundai.jpg")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("35")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("13")
    itm.addObject("Kia")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("kia.jpg")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("29")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("14")
    itm.addObject("Mercedes")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("mercedes.jpg")
    itm.addObject("Nice!")
    itm.addObject("")
    itm.addObject("20")
    itm.addObject("0")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("15")
    itm.addObject("Peugeot")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("peugeot.jpg")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("100")
    itm.addObject("1")
    dsource.addObject(itm)
    itm = NSMutableArray()
    itm.addObject("16")
    itm.addObject("Renault")
    itm.addObject("0")
    itm.addObject("0")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("")
    itm.addObject("false")
    itm.addObject("renault.jpg")
    itm.addObject("from France")
    itm.addObject("")
    itm.addObject("100")
    itm.addObject("1")
    dsource.addObject(itm)
    return dsource
    }
    
    // 3. JSON file DataSource
    func generateJSONfileDataSource() -> NSMutableArray {
        var dsource = NSMutableArray()
        var filePath = NSBundle.mainBundle().pathForResource("Cars", ofType: "json")
        var fileContents = NSData(contentsOfFile: filePath!, options: .DataReadingMappedIfSafe, error: nil)
        var opt: NSJSONReadingOptions = NSJSONReadingOptions.MutableContainers
        var err: NSErrorPointer = nil
        let SourceArr: NSArray = NSJSONSerialization.JSONObjectWithData(fileContents!, options: opt, error: err) as NSArray
        for row in SourceArr {
            var itm = NSMutableArray()
            itm.addObject(row["ID"] as NSString)
            itm.addObject(row["Corporation"] as NSString)
            itm.addObject(row["CarID"] as NSString)
            itm.addObject(row["ModelID"] as NSString)
            itm.addObject(row["DateFirst"] as NSString)
            itm.addObject(row["TimeFirst"] as NSString)
            itm.addObject(row["Phone"] as NSString)
            itm.addObject(row["Mark"] as NSString)
            itm.addObject(row["Picture"] as NSString)
            itm.addObject(row["Remark"] as NSString)
            itm.addObject(row["ZipCode"] as NSString)
            itm.addObject(row["Availability"] as NSString)
            itm.addObject(row["Accessibility"] as NSString)
            dsource.addObject(itm)
        }
        return dsource
    }
    
    // 4. XML file DataSource
    func generateXMLfileDataSource() -> NSMutableArray {
        var dsource = NSMutableArray()
        var filePath = NSBundle.mainBundle().pathForResource("Cars", ofType: "xml")
        var fileContents: NSString = NSString(contentsOfFile: filePath!, encoding:NSUTF8StringEncoding, error: nil)!
        var dict: NSDictionary = NSDictionary(object: NSDictionary(object: XMLReader.XMLtoDictionary(fileContents, error: nil), forKey:"table"), forKey:"Cars")
        var dict0: NSDictionary = dict["Cars"] as NSDictionary
        for (rKey, rValue) in dict0 {
            for (key, value) in rValue as NSDictionary {
                if key as NSString == "table" {
                for (key0, val0) in value as NSDictionary {
                    if key0 as NSString == "Cars" {
                        for row in val0 as NSArray {
                            var arr = [Int: NSString]()
                            for (key1, val1) in row as NSDictionary {
                                if key1 as NSString == "ID" { arr[0] = (val1["text"] as NSString) }
                                if key1 as NSString == "Corporation" { arr[1] = (val1["text"] as NSString) }
                                if key1 as NSString == "CarID" { arr[2] = (val1["text"] as NSString) }
                                if key1 as NSString == "ModelID" { arr[3] = (val1["text"] as NSString) }
                                if key1 as NSString == "DateFirst" { arr[4] = (val1["text"] as NSString) }
                                if key1 as NSString == "TimeFirst" { arr[5] = (val1["text"] as NSString) }
                                if key1 as NSString == "Phone" { arr[6] = (val1["text"] as NSString) }
                                if key1 as NSString == "Mark" { arr[7] = (val1["text"] as NSString) }
                                if key1 as NSString == "Picture" { arr[8] = (val1["text"] as NSString) }
                                if key1 as NSString == "Remark" { arr[9] = (val1["text"] as NSString) }
                                if key1 as NSString == "ZipCode" { arr[10] = (val1["text"] as NSString) }
                                if key1 as NSString == "Availability" { arr[11] = (val1["text"] as NSString) }
                                if key1 as NSString == "Accessibility" { arr[12] = (val1["text"] as NSString) }
                            }
                            var itm = NSMutableArray()
                            for var i = 0; i < arr.count; i++ {
                                itm.addObject(arr[i]! == "nil" ? "" : arr[i]!)
                            }
                            dsource.addObject(itm)
                        }
                    }
                    }
                }
            }
        }
        return dsource
    }
    
    // Format Phone Number in TextField cell
    func phoneNumberFormat(initNumber: NSString) -> NSString {
        if initNumber.length > 9 {
            if "0123456789".rangeOfString(initNumber.substringWithRange(NSRange(location: 1, length: 1))) != nil {
                var strNumber: NSMutableString = initNumber.mutableCopy() as NSMutableString
                strNumber.insertString("(", atIndex: 0)
                strNumber.insertString(")", atIndex: 4)
                strNumber.insertString("-", atIndex: 5)
                strNumber.insertString("-", atIndex: 9)
                return strNumber.copy() as NSString
            }
        }
        return initNumber
    }
    
    // Show message box
    func msgbox(msg: NSString, title: NSString) {
        let alert = UIAlertView(title: title, message:msg, delegate: self, cancelButtonTitle: "Ok", otherButtonTitles: "", "")
        alert.show()
    }

}

